-- ডাটাবেজ তৈরি
CREATE DATABASE IF NOT EXISTS bd_club;
USE bd_club;

-- ইউজার টেবিল
CREATE TABLE users (
    id INT PRIMARY KEY AUTO_INCREMENT,
    uiid VARCHAR(20) UNIQUE NOT NULL,
    name VARCHAR(100) NOT NULL,
    phone VARCHAR(20) UNIQUE NOT NULL,
    password VARCHAR(255) NOT NULL,
    email VARCHAR(100),
    deposit_balance DECIMAL(10,2) DEFAULT 0.00,
    winning_balance DECIMAL(10,2) DEFAULT 0.00,
    total_referrals INT DEFAULT 0,
    total_bonus DECIMAL(10,2) DEFAULT 0.00,
    referral_code VARCHAR(20),
    referrer_id INT,
    has_deposited BOOLEAN DEFAULT FALSE,
    status ENUM('active', 'suspended', 'blocked') DEFAULT 'active',
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    last_login DATETIME,
    INDEX idx_uiid (uiid),
    INDEX idx_phone (phone),
    INDEX idx_status (status),
    FOREIGN KEY (referrer_id) REFERENCES users(id) ON DELETE SET NULL
);

-- অ্যাডমিন টেবিল
CREATE TABLE admins (
    id INT PRIMARY KEY AUTO_INCREMENT,
    username VARCHAR(50) UNIQUE NOT NULL,
    password VARCHAR(255) NOT NULL,
    email VARCHAR(100),
    role ENUM('superadmin', 'admin', 'moderator') DEFAULT 'admin',
    status ENUM('active', 'inactive') DEFAULT 'active',
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    last_login DATETIME
);

-- ডিপোজিট টেবিল
CREATE TABLE deposits (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    uiid VARCHAR(20) NOT NULL,
    amount DECIMAL(10,2) NOT NULL,
    method ENUM('bKash', 'Nagad', 'Rocket') NOT NULL,
    agent_number VARCHAR(20) NOT NULL,
    transaction_id VARCHAR(100) NOT NULL,
    status ENUM('pending', 'approved', 'rejected') DEFAULT 'pending',
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    approved_at DATETIME,
    approved_by INT,
    notes TEXT,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (approved_by) REFERENCES admins(id) ON DELETE SET NULL,
    INDEX idx_user_id (user_id),
    INDEX idx_status (status),
    INDEX idx_created_at (created_at)
);

-- উইথড্র টেবিল
CREATE TABLE withdrawals (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    uiid VARCHAR(20) NOT NULL,
    amount DECIMAL(10,2) NOT NULL,
    method VARCHAR(50) NOT NULL,
    account_number VARCHAR(50) NOT NULL,
    account_name VARCHAR(100),
    status ENUM('pending', 'approved', 'rejected') DEFAULT 'pending',
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    approved_at DATETIME,
    approved_by INT,
    notes TEXT,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (approved_by) REFERENCES admins(id) ON DELETE SET NULL,
    INDEX idx_user_id (user_id),
    INDEX idx_status (status)
);

-- বেট টেবিল
CREATE TABLE bets (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    bet_id VARCHAR(50) UNIQUE NOT NULL,
    bet_type VARCHAR(20) NOT NULL,
    amount DECIMAL(10,2) NOT NULL,
    period VARCHAR(50) NOT NULL,
    result ENUM('pending', 'win', 'loss') DEFAULT 'pending',
    win_amount DECIMAL(10,2) DEFAULT 0.00,
    multiplier DECIMAL(5,2),
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user_id (user_id),
    INDEX idx_result (result),
    INDEX idx_created_at (created_at)
);

-- গেম রেজাল্ট টেবিল
CREATE TABLE game_results (
    id INT PRIMARY KEY AUTO_INCREMENT,
    period VARCHAR(50) UNIQUE NOT NULL,
    number INT NOT NULL,
    color VARCHAR(20) NOT NULL,
    size VARCHAR(10) NOT NULL,
    result_time DATETIME DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_period (period),
    INDEX idx_result_time (result_time)
);

-- গিফট কোড টেবিল
CREATE TABLE gift_codes (
    id INT PRIMARY KEY AUTO_INCREMENT,
    code VARCHAR(50) UNIQUE NOT NULL,
    amount DECIMAL(10,2) NOT NULL,
    status ENUM('active', 'redeemed', 'expired', 'deactivated') DEFAULT 'active',
    created_by INT,
    redeemed_by INT,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    redeemed_at DATETIME,
    expiry_date DATE,
    notes TEXT,
    FOREIGN KEY (created_by) REFERENCES admins(id) ON DELETE SET NULL,
    FOREIGN KEY (redeemed_by) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_code (code),
    INDEX idx_status (status)
);

-- পেমেন্ট নাম্বার টেবিল
CREATE TABLE payment_numbers (
    id INT PRIMARY KEY AUTO_INCREMENT,
    number VARCHAR(20) NOT NULL,
    label VARCHAR(100) NOT NULL,
    type ENUM('bKash', 'Nagad', 'Rocket') NOT NULL,
    is_active BOOLEAN DEFAULT TRUE,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_type (type),
    INDEX idx_is_active (is_active)
);

-- সেটিংস টেবিল
CREATE TABLE settings (
    id INT PRIMARY KEY AUTO_INCREMENT,
    setting_key VARCHAR(100) UNIQUE NOT NULL,
    setting_value TEXT,
    description TEXT,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_setting_key (setting_key)
);

-- ট্রানজেকশন হিস্ট্রি ভিউ
CREATE VIEW transaction_history AS
SELECT 
    'deposit' as transaction_type,
    id,
    user_id,
    uiid,
    amount,
    status,
    created_at,
    CONCAT('Deposit via ', method) as description
FROM deposits
UNION ALL
SELECT 
    'withdrawal' as transaction_type,
    id,
    user_id,
    uiid,
    amount,
    status,
    created_at,
    CONCAT('Withdrawal to ', method) as description
FROM withdrawals
ORDER BY created_at DESC;

-- ইনিশিয়াল ডাটা ইনসার্ট
INSERT INTO admins (username, password, email, role) VALUES 
('admin', '$2y$10$YourHashedPasswordHere', 'admin@bdclub.com', 'superadmin');

INSERT INTO payment_numbers (number, label, type) VALUES
('017XXXXXXXX', 'এজেন্ট ১ (বিকাশ)', 'bKash'),
('018XXXXXXXX', 'এজেন্ট ২ (নগদ)', 'Nagad'),
('019XXXXXXXX', 'এজেন্ট ৩ (রকেট)', 'Rocket');

INSERT INTO settings (setting_key, setting_value, description) VALUES
('site_name', 'BD CLUB', 'সাইটের নাম'),
('site_url', 'http://localhost/bd-club/', 'সাইট URL'),
('admin_email', 'admin@bdclub.com', 'অ্যাডমিন ইমেইল'),
('welcome_bonus', '0', 'নতুন ইউজার বোনাস'),
('referral_bonus', '0', 'রেফারেল বোনাস'),
('min_deposit', '100', 'ন্যূনতম ডিপোজিট'),
('min_withdraw', '500', 'ন্যূনতম উইথড্র'),
('notice', 'স্বাগতম BD CLUB! ডিপোজিট করুন এবং জিতুন!', 'নোটিশ বার মেসেজ'),
('support_link', 'https://t.me/bdclub_support', 'সাপোর্ট গ্রুপ লিংক'),
('game_timer', '60', 'গেম টাইমার (সেকেন্ড)'),
('maintenance_mode', '0', 'মেইনটেনেন্স মোড (1=চালু, 0=বন্ধ)');

-- স্যাম্পল ইউজার (পাসওয়ার্ড: 123456)
INSERT INTO users (uiid, name, phone, password, deposit_balance, winning_balance) VALUES
('BD100001', 'টেস্ট ইউজার', '01710000001', '$2y$10$4Hk1V7e6HwQa8wZ9XyZzUeBvC1D2E3F4G5H6I7J8K9L0M1N2O3P4Q', 1000.00, 500.00);

-- স্যাম্পল গেম রেজাল্ট
INSERT INTO game_results (period, number, color, size) VALUES
('100001', 7, 'green', 'big'),
('100002', 2, 'red', 'small'),
('100003', 5, 'violet', 'small'),
('100004', 8, 'red', 'big'),
('100005', 1, 'green', 'small');